function data = PlotPlanetNNRandCgraphs(data)

% Function to produce the R and C graphs for the Nearest Neightbour statistic summary
% for Tab 3

% Author: Ciaran Beggan
% Created: 19-11-2010
% Last Modified: 20-11-2011

% Version 0.4: Removed the k=1 and k=2 choices for the planetary NN; Added
%               third-planet option
% Version 0.3: Added in more detailed C and R curves
% Version 0.2: Added the option for drawing NN scavenged k = 1 and k = 2
%              curves
% Version 0.1: Initial script produce plots and draw
%              points into the relevant axes

% Note: the data generated here comes from a script I wrote called Baloga
% simulations which derives these values from 1000 simulations of a Poisson
% distributed image

% store = [mean(C) std(C) mean(R) std(R) N]


pixsize = 1;
stat = data.distances*pixsize;
data.nNN = length(stat);
ra = mean(stat);

% Load the correct store of bias curves for Poisson NN

if  strcmp(data.PlanetDatasetSize, 'Whole')
    rho_0 = data.NN/(ellarea(data.PlanetRadius(1),data.PlanetRadius(2)));
    NNbias_fullsphere_k_0 = dlmread('skdata\NNbias_fullsphere_k_0.dat');
    store =  NNbias_fullsphere_k_0(:, [4, 5, 2, 3, 1]);
elseif strcmp(data.PlanetDatasetSize, 'Half')
    NNbias_halfsphere_k_0 = dlmread('skdata\NNbias_halfsphere_k_0.dat');
    store =  NNbias_halfsphere_k_0(:, [4, 5, 2, 3, 1]);
    rho_0 = data.NN/(ellarea(data.PlanetRadius(1),data.PlanetRadius(2))/2);
elseif  strcmp(data.PlanetDatasetSize, 'Third')
    NNbias_thirdsphere_k_0 = dlmread('skdata\NNbias_thirdsphere_k_0.dat');
    store =  NNbias_thirdsphere_k_0(:, [4, 5, 2, 3, 1]);
    rho_0 = data.NN/(ellarea(data.PlanetRadius(1),data.PlanetRadius(2))/3);
elseif strcmp(data.PlanetDatasetSize, 'Quarter')
    NNbias_quartersphere_k_0 = dlmread('skdata\NNbias_quartersphere_k_0.dat');
    store =  NNbias_quartersphere_k_0(:, [4, 5, 2, 3, 1]);
    rho_0 = data.NN/(ellarea(data.PlanetRadius(1),data.PlanetRadius(2))/4);
else
    NNbias_flat_k_0 = dlmread('skdata\NNbias_flat_k_0.dat');
    store =  NNbias_flat_k_0(:, [4, 5, 2, 3, 1]);
    rho_0 = data.NN/(data.chull_area*pixsize^2);
end



% Calculate R and C

re = 1/(2* sqrt(rho_0));  data.re = re;
sde = 0.26136/( sqrt(data.nNN*rho_0));  data.sde = sde;
c = (ra - re)/sde;   data.c = c;
r = ra/re;           data.r = r;


% Plot confidence limits on R
set(gcbf,'CurrentAxes',findobj(gcbf,'Tag','Axes17'));    % move to the correct axes
cla reset

Xr = reshape(repmat(store(:,5), 1,2)', length(store)*2,1);
Yr = reshape(repmat(store(:,4), 1,2)', length(store)*2,1);
Y2r = reshape(repmat(store(:,4)*2, 1,2)', length(store)*2,1); % 2 sigma limits
R = reshape(repmat(store(:,3), 1,2)', length(store)*2,1);

hold on
patchthreshclip = interp1(store(:,5), store(:,4),data.nNN+100,'linear');
Rthreshclip = interp1(store(:,5), store(:,3),data.nNN+100,'linear');
rows = find(Xr < data.nNN+100);
Xrclip = [Xr(rows) ; data.nNN + 100; data.nNN + 100;10];
Yrclip = [Yr(rows) ; patchthreshclip; 0 ; 0];
Y2rclip = [Y2r(rows) ; patchthreshclip*2; 0 ; 0];
Rclip = [R(rows); Rthreshclip; 1; 1];

patch(Xrclip,  Y2rclip+Rclip, [0.7 0.7 0.7], 'EdgeColor', 'none', 'Clipping', 'On')
patch(Xrclip, -Y2rclip+Rclip, [0.7 0.7 0.7], 'EdgeColor', 'none', 'Clipping', 'On')
patch(Xrclip,   Yrclip+Rclip, [0.4 0.4 0.4], 'EdgeColor', 'none', 'Clipping', 'On')
patch(Xrclip, - Yrclip+Rclip, [0.4 0.4 0.4], 'EdgeColor', 'none', 'Clipping', 'On')
plot([store(:,5); 10000], [store(:,3); 1], 'k', 'LineWidth', 1)

plot(data.nNN, r, 'd', 'MarkerSize', 8, 'MarkerFaceColor', 'r')
%axis([10 data.nNN+100 0 2])
xlim([10 data.nNN+100])
set(gca,'Tag','Axes17');
set(gcbf,'Userdata',data)

% C
set(gcbf,'CurrentAxes',findobj(gcbf,'Tag','Axes18'));    % move to the correct axes
cla reset

Xc = reshape(repmat(store(:,5), 1,2)', length(store)*2,1);
Yc = reshape(repmat(store(:,2), 1,2)', length(store)*2,1);
Y2c = reshape(repmat(store(:,2)*2, 1,2)', length(store)*2,1); % 2 sigma limits
C = reshape(repmat(store(:,1), 1,2)', length(store)*2,1);

patchthreshclip = interp1(store(:,5), store(:,4),data.nNN+100,'linear');
Cthreshclip = interp1(store(:,5), store(:,3),data.nNN+100,'linear');
rows = find(Xc < data.nNN+100);
Xcclip = [Xc(rows) ; data.nNN + 100; data.nNN + 100;10];
Ycclip = [Yc(rows) ; patchthreshclip; 0 ; 0];
Y2cclip = [Y2c(rows) ; patchthreshclip*2; 0 ; 0];
Cclip = [C(rows); Cthreshclip; 0; 0];


hold on
patch(Xcclip, Y2cclip + Cclip, [0.7 0.7 0.7], 'EdgeColor', 'none')
patch(Xcclip, -Y2cclip + Cclip, [0.7 0.7 0.7], 'EdgeColor', 'none')
patch(Xcclip, Ycclip + Cclip, [0.4 0.4 0.4], 'EdgeColor', 'none')
patch(Xcclip, -Ycclip + Cclip, [0.4 0.4 0.4], 'EdgeColor', 'none')
plot([store(:,5); 10000], [store(:,1); 0], 'k', 'LineWidth', 1)

plot(data.nNN, c, 'd', 'MarkerSize', 8, 'MarkerFaceColor', 'r')
axis([10 data.nNN+100 -2-abs(c) 4+abs(c)])
set(gca,'Tag','Axes18');
set(gcbf,'Userdata',data)


% R and C statistics
set(findobj(gcbf,'Tag','PlanetCstatisticOut'),'String',num2str(data.c, '%5.3f'));
set(findobj(gcbf,'Tag','PlanetRstatisticOut'),'String',num2str(data.r, '%5.3f'));

% Interpolate to find an approximate limit

if data.NN > 9
    data.cthreshneg1 = interp1([store(:,5); 100001], [store(:,1)-1*store(:,2); -2],data.nNN,'linear');
    data.cthreshneg2 = interp1([store(:,5); 100001], [store(:,1)-2*store(:,2); -2],data.nNN,'linear');
    set(findobj(gcbf,'Tag','PlanetCThresholdOutNeg1s'),'String',num2str(data.cthreshneg1, '%5.3f'));
    set(findobj(gcbf,'Tag','PlanetCThresholdOutNeg2s'),'String',num2str(data.cthreshneg2, '%5.3f'));
    data.cthreshpos1 = interp1([store(:,5); 100001], [store(:,1)+1*store(:,2); 2],data.nNN,'linear');
    data.cthreshpos2 = interp1([store(:,5); 100001], [store(:,1)+2*store(:,2); 2],data.nNN,'linear');
    set(findobj(gcbf,'Tag','PlanetCThresholdOutPos1s'),'String',num2str(data.cthreshpos1, '%5.3f'));
    set(findobj(gcbf,'Tag','PlanetCThresholdOutPos2s'),'String',num2str(data.cthreshpos2, '%5.3f'));

    data.rthreshneg1 = interp1([store(:,5); 100001], [store(:,3)-1*store(:,4); 1],data.nNN,'linear');
    data.rthreshneg2 = interp1([store(:,5); 100001], [store(:,3)-2*store(:,4); 1],data.nNN,'linear');
    set(findobj(gcbf,'Tag','PlanetRThresholdOutNeg1s'),'String',num2str(data.rthreshneg1, '%5.3f'));
    set(findobj(gcbf,'Tag','PlanetRThresholdOutNeg2s'),'String',num2str(data.rthreshneg2, '%5.3f'));
    data.rthreshpos1 = interp1([store(:,5); 100001], [store(:,3)+1*store(:,4); 1],data.nNN,'linear');
    data.rthreshpos2 = interp1([store(:,5); 100001], [store(:,3)+2*store(:,4); 1],data.nNN,'linear');
    set(findobj(gcbf,'Tag','PlanetRThresholdOutPos1s'),'String',num2str(data.rthreshpos1, '%5.3f'));
    set(findobj(gcbf,'Tag','PlanetRThresholdOutPos2s'),'String',num2str(data.rthreshpos2, '%5.3f'));
else
    set(findobj(gcbf,'Tag','PlanetCThresholdOutPos'),'String','-');
    set(findobj(gcbf,'Tag','PlanetRThresholdOutPos'),'String','-');
end

% Work out the implication of the R and c values i.e. random, clustered,
% uniform, significant
set(findobj(gcbf,'Tag','ImplicationRp'),'String',' ');
set(findobj(gcbf,'Tag','ImplicationCp'),'String',' ');

if  (data.r < data.rthreshneg2)
    set(findobj(gcbf,'Tag','ImplicationRp'),'String','Clustered (R < -2std)');
elseif (data.r > data.rthreshneg2) &&  (data.r < data.rthreshneg1)
    set(findobj(gcbf,'Tag','ImplicationRp'),'String','Random (-2std < R < -1std)');
elseif (data.r > data.rthreshneg1) &&  (data.r < data.rthreshpos1)
    set(findobj(gcbf,'Tag','ImplicationRp'),'String','Random (-1std < R < +1std)');
elseif (data.r > data.rthreshpos1) &&  (data.r < data.rthreshpos2)
    set(findobj(gcbf,'Tag','ImplicationRp'),'String','Random (+1std < R < +2std)');
elseif (data.r > data.rthreshpos2)
    set(findobj(gcbf,'Tag','ImplicationRp'),'String','Uniform (R > +2td)');
else
    set(findobj(gcbf,'Tag','ImplicationRp'),'String','Unknown outcome');
end

if (data.r > data.rthreshneg2) &&  (data.r < data.rthreshpos2)
    if  (data.c < data.cthreshneg2)
        set(findobj(gcbf,'Tag','ImplicationCp'),'String','Not Significant (c < -2std)');
    elseif (data.c > data.cthreshneg2) &&  (data.c < data.cthreshneg1)
        set(findobj(gcbf,'Tag','ImplicationCp'),'String','Significant (-2std < c < -1std)');
    elseif (data.c > data.cthreshneg1) &&  (data.c < data.cthreshpos1)
        set(findobj(gcbf,'Tag','ImplicationCp'),'String','Significant (-1std < c < +1std)');
    elseif (data.c > data.cthreshpos1) &&  (data.c < data.cthreshpos2)
        set(findobj(gcbf,'Tag','ImplicationCp'),'String','Significant (+1std < c < +2std)');
    elseif (data.c > data.cthreshpos2)
        set(findobj(gcbf,'Tag','ImplicationCp'),'String','Not significant (c > +2std)');
    end
else
    set(findobj(gcbf,'Tag','ImplicationCp'),'String','Not applicable');
end

% Ideal values of R and C
data.idealc = interp1([store(:,5); 100001], [store(:,1); 0],data.nNN,'linear');
data.idealr = interp1([store(:,5); 100001], [store(:,3); 1],data.nNN,'linear');