function gias_guifun_v06(action)
% GUI functions for GIAS GUI
%
% Adapted from fm_guifun written by Adam Wilmer, aiw99r@ecs.soton.ac.uk
% Controls the functions that are called by GIAS2.0 GUI
%
% Version: 0.6
% Created: 18-11-2010
%
% Last Updated: 28-11-2011
%
% Author: Ciaran Beggan (University of Edinburgh/ British Geological Survey)
%
% Version 0.1:  Counts the bubbles in an image, outputs the c, r and skew
%               vs. kurtosis and area histograms
%
% Version 0.2: A more sophistication control system for the GUI, including
%              separate tabs etc for the output
%
% Version 0.21: Pre-release version
% Version 0.22: Sliders in the input area removed
%
% Version 0.31: All references relating to the Logistic casd have been
%               removed
% Version 0.32: Updated to allow the user to enter the area of the Hull
%               directly
%
% Version 0.4: Added a third tab to allow planetary bodies to be analysed
% Version 0.41: Made adjustments to the loading file button, the convex
%               pixel suze units and the planetary NN tab
%
% Version 0.5: Added in the calls for the planetary bodies to be analysed.
%               There is a large amount of code copied from the NN analysis
%               tab, but it has to be relabelled, as different codes are
%               computed. Planets now callable by name from ArcGIS file.
%
% Version 0.6: Load Planetary data using a GUI file open dialog box. Save
%               output results to a file using an GUI file SaveAs dialog
%               box. Select the pixel size from a drop down box.

global data
switch(action)

    % Init  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    case('create')
        data.pathname = '';   % path pointing to input data folder
        data.pathname_outfile = '';   % path pointing to output data folder
        data.pathname_proj = '';   % path pointing to projection data folder

        data.outfile = 'Output';        % Output filename (default)

        % ------------------- window handle storage --------------------
        data.hmain = get(findobj(gcbf,'Tag','Fig1'));
        data.image_processed = 0;           % Is there any imagery processed yet? Update if a new image
        data.centroid = [];
        data.areas = [];
        data.chull_area = [];
        data.perimeters = [];
        data.equiv_diameters = [];
        data.orientations = [];
        data.eccentricity = [];
        data.NNskew = 0;
        data.NNkurt = 0;


        % --------- IMAGE 1 ---------------------------------------------------
        %

        data.input1 = [];   % image data
        data.filename1 = [];    % filename corresponding to image 1
        data.isimage = 1;
        data.NNonlyimage = 0;
        data.NNonlylist = 0;
        data.invertimage = 0;
        data.PlanetaryData = 0;
        data.PlanetRadius = []; %Empty by default

        % -------- SOME PARAMETERS ------------------------
        data.dispText = 0;
        data.PixelSize = 1;                 % Physical Pixel size
        data.PixelUnit = 'm';
        data.IgnoreBoundaryBubbles = 1;
        data.IgnoreSmallArea = 10;
        data.SetThresholdLevelUpper = 250;
        data.SetThresholdLevelLower = 0;
        data.IgnoreLargeArea = 'Max';
        data.SetNormPoissonThresholdLevel = 0;
        data.CheckInputHullArea = 0;
        data.InputHullArea = 0;

        % Bubble Size Frequency Options
        data.BubbleBinSizeOption  = 'Phi';  % phi scale or Custom
        data.BubbleBinSize = 500;
        data.BubbleXAxisOption = 'Default'; data.BubbleXAxisMinSize = 100; data.BubbleXAxisMaxSize = 100000;
        data.BubbleYAxisOption = 'Linear';
        data.BubbleYMaxAxisOption = 0; data.BubbleYAxisMaxSize = 100;

        % Nearest Neigbour Frequency Options
        data.NNModelSelection = 'Poisson';  data.setkvalue = 1;
        data.NNBinSizeOption = 'Default';  % Default(log) or Custom (linear)
        data.NNBinSize = 5;
        data.NNXAxisOption = 'Default'; data.NNXAxisMinSize = 0; data.NNXAxisMaxSize = 50;
        data.NNYAxisOption = 'Linear';
        data.NNYMaxAxisOption = 0; data.NNYAxisMaxSize = 100;
        
        % Planetary Nearest Neigbour Frequency Options
        data.PlanetDatasetSize = 'Whole';
        data.PlanetNNModelSelection = 'Poisson';  data.Planetsetkvalue = 1;
        data.PlanetNNBinSizeOption = 'Default';  % Default(log) or Custom (linear)
        data.PlanetNNBinSize = 50;
        data.PlanetNNXAxisOption = 'Default'; data.PlanetNNXAxisMinSize = 0; data.PlanetNNXAxisMaxSize = 50;
        data.PlanetNNYAxisOption = 'Linear';
        data.PlanetNNYMaxAxisOption = 0; data.NNYAxisMaxSize = 100;
        
        % Bubble Perimeter
        data.BubPerimeter = 'Histogram';

        % Summary Statistics
        data.SummaryStatistics = 'Area';

        set(gcbf,'Userdata',data);


        % Load image or 2-column data file  %%
    case('loadsem')
        dispText('','b');
        data = get(gcbf,'Userdata');
        pathname 	= data.pathname;


        dispTag('SEM_im','r');     % this stuff isn't that apparent in the application??!!
        dispText('Loading image 1','b');

        if data.NNonlylist || data.PlanetaryData
            [filename, pathname] = uigetfile([pathname '*.*'], 'Load Centroid List File ');    % GUI file browser
            if filename~=0         % if we have a file
                [M1] = dlmread([pathname, filename]);

                data.input1 = M1;
                set(gcbf,'Userdata',data);
                
                data.isimage = 0;   % Switch off the default assumption it is an image

                data.pathname = pathname;   % save pathname of this file

                data.filename1 = filename;

                dispTag('SEM_im','k');
                dispText('','b');
                data.image_processed = 0;
                set(gcbf,'Userdata',data);
            end;

        else
            [filename, pathname] = uigetfile([pathname '*.*'], 'Load image/data file');    % GUI file browser
            if filename~=0         % if we have a file
                % test if file is a text file or an image file
                fileid = filename(end-2:end);
                if (strmatch(fileid, ['tif'; 'jpg'; 'bmp']))

                    [M1] = imread([pathname, filename]);

                    data.input1 = M1;
                    set(gcbf,'Userdata',data);

                    data = get(gcbf,'Userdata');

                    data.pathname = pathname;   % save pathname of this file
                    data.isimage = 1;
                    data.filename1 = filename;
                    data.NNonlylist = 0;
                    dispTag('SEM_im','k');
                    dispText('','b');
                    data.image_processed = 0;
                    set(gcbf,'Userdata',data);

                elseif (strmatch(fileid, ['txt'; 'dat']))
                    [M1] = dlmread([pathname, filename]);

                    data.input1 = M1;
                    set(gcbf,'Userdata',data);


                    data.pathname = pathname;   % save pathname of this file

                    data.filename1 = filename;
                    if ~data.PlanetaryData
                        data.NNonlylist = 1;  
                        data.isimage = 0;  
                    end
                    dispTag('SEM_im','k');
                    dispText('','b');
                    data.image_processed = 0;
                    set(gcbf,'Userdata',data);

                else  % Assuming we fall through both statements let's assume it's an image
                    [M1] = imread([pathname, filename]);

                    data.input1 = M1;
                    set(gcbf,'Userdata',data);
                    data.isimage = 1;
                    data = get(gcbf,'Userdata');

                    data.pathname = pathname;   % save pathname of this file

                    data.filename1 = filename;
                    
                    data.NNonlylist = 0;

                    dispTag('SEM_im','k');
                    dispText('','b');
                    data.image_processed = 0;
                    set(gcbf,'Userdata',data);
                end
            end;

        end

    case('input1Select')
        data = get(gcbf,'Userdata');

        if ~isempty(data.input1) % Check if a file has been input
            set(gcbf,'CurrentAxes',findobj(gcbf,'Tag','Axes1'));    % move to the correct axes
            cla reset;       % clear the image corresponding to these axes

            mx = data.input1;

            imshow(mx, 'InitialMagnification',100)
            imagesc(mx(:,:,1));   % display image
            axis image;   % fit axis box tightly around image
            %axis ij;      % puts MATLAB into its "matrix" axes mode.  The coordinate system origin is at the upper left corner.  The i axis is vertical and is numbered from top to bottom.  The j axis is horizontal and is numbered from left to right.
            axis off;   % turns off labelling
            colormap gray

            set(gca,'Tag','Axes1');
        end
        set(gcbf,'Userdata',data);

    case('setOutputName')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','textFilename'),'String');
        data.outfile = t;

        set(gcbf,'Userdata',data);

        % case('setPixelSize')
        %     data = get(gcbf,'Userdata');
        %     t = get(findobj(gcbf,'Tag','setPixelSize'),'String');
        %     data.PixelSize = str2double(t);
        %
        %     set(gcbf,'Userdata',data);
        %

    case('setInvertImage')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','InvertImage'),'Value');
        if t == 1
            data.invertimage = 1;
        else
            data.invertimage = 0;
        end

        set(gcbf,'Userdata',data);

    case('setPlanetaryDataOnly')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','PlanetaryData'),'Value');
        if t == 1
            data.PlanetaryData = 1; 
            data.NNonlylist = 0;  % Disable the NN only list option
            
        else
            data.PlanetaryData = 0;
            data = get(gcbf,'Userdata');
            set(findobj(gcbf,'Tag','PlanetName'),'String', ' ');
            set(findobj(gcbf,'Tag','PlanetUnit'),'String', ' ');

            return
        end
        
        pathname 	= data.pathname_proj;
        [filename, pathname] = uigetfile([pathname '*.*'], 'Load planetary Projection data ');    % GUI file browser
        if filename~=0         % if we have a file
            
            prj = textread(fullfile(pathname,filename), '%s', 'delimiter', ',');
            meanradius = str2double(prj{4});
            flattening = str2double(prj{5}(1:end-2));
            if (flattening==0)
                data.PlanetRadius = [meanradius, meanradius]./1000;
            else
                data.PlanetRadius = [meanradius, meanradius*(1-1/flattening)]./1000;
            end
            
            % Add piece of code to place the file name into an output box
            % to confirm loading of projection data
            set(findobj(gcbf,'Tag','PlanetName'),'String', [filename(1:end-4) ' loaded']);
            set(findobj(gcbf,'Tag','PlanetUnit'),'String', 'Input units: degrees; hull area: m^2');
        else
            set(findobj(gcbf,'Tag','PlanetName'),'String', 'NO FILE loaded');

        end
        
        % Save the projection folder data
        data.pathname_proj = pathname;
        set(gcbf,'Userdata',data);
        

    case('setIgnoreBoundaryBubbles')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','IgnoreBoundaryBubbles'),'Value');
        if t == 1
            data.IgnoreBoundaryBubbles = 1;
        else
            data.IgnoreBoundaryBubbles = 0;
        end

        set(gcbf,'Userdata',data);

    case('setCheckInputHullArea')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','CheckInputHullArea'),'Value');
        if t == 1
            data.CheckInputHullArea = 1;
            t1 = get(findobj(gcbf,'Tag','textInputHullArea'),'String');
            if isnan(str2double(t1))
                set(findobj(gcbf,'Tag','textInputHullArea'),'String','Enter Value');
                drawnow, pause(0.35)
                set(findobj(gcbf,'Tag','textInputHullArea'),'String','Enter Hull Area');
            else
                data.InputHullArea = floor(str2double(t1));
            end
        else
            data.CheckInputHullArea = 0;
        end

        set(gcbf,'Userdata',data);

    case('setInputHullArea')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','CheckInputHullArea'),'Value');
        if t == 1
            t1 = get(findobj(gcbf,'Tag','textInputHullArea'),'String');
            if isnan(str2double(t1))
                set(findobj(gcbf,'Tag','textInputHullArea'),'String','Enter Value');
                drawnow, pause(0.65)
                set(findobj(gcbf,'Tag','textInputHullArea'),'String','Enter Hull Area');
            else
                data.InputHullArea = floor(str2double(t1));
            end

        else
            data.CheckInputHullArea = 0;
        end

        set(gcbf,'Userdata',data);



    case('setIgnoreSizeLevel')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','IgnoreSizeLevel'),'String');
        data.IgnoreSmallArea = floor(str2double(t));

        set(gcbf,'Userdata',data);

    case('setIgnoreMaxSizeLevel')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','IgnoreMaxSizeLevel'),'String');

        if (strcmp(t, 'Max'))
            data.IgnoreLargeArea = 'Max';
        else
            data.IgnoreLargeArea = floor(str2double(t));
            data.BubbleXAxisMaxSize = floor(str2double(t));
        end
        set(gcbf,'Userdata',data);


    case('setThresholdLevel')
        data = get(gcbf,'Userdata');
        tu = get(findobj(gcbf,'Tag','DetectionThresholdLevelUpper'),'String');
        if str2double(tu) > 254
            data.SetThresholdLevelUpper = 254;
        else
            data.SetThresholdLevelUpper = floor(str2double(tu));
        end

        tl = get(findobj(gcbf,'Tag','DetectionThresholdLevelLower'),'String');
        if str2double(tl) < 0
            data.SetThresholdLevelLower = 0;
        else
            data.SetThresholdLevelLower = floor(str2double(tl));
        end


        %set(findobj(gcbf,'Tag','textThresholdslider'),'String',num2str(data.SetThresholdLevel));

        set(gcbf,'Userdata',data);


    case('setNormPoissonThresholdLevel')
        data = get(gcbf,'Userdata');
        % Have to use the correct units (as selected in the Units area)
        t = get(findobj(gcbf,'Tag','NormPoissonThresholdLevel'),'String');
        if t < 0
            data.SetNormPoissonThresholdLevel = 0 ;
        else
            data.SetNormPoissonThresholdLevel = str2double(t);

        end

        set(gcbf,'Userdata',data);
        
       
    case('setNNPixelListonly')
        data = get(gcbf,'Userdata');

        t = get(findobj(gcbf,'Tag','NNCentroidListonly'),'Value');
        if t == 1
            data.NNonlylist = 1;
            set(findobj(gcbf,'Tag','NNCentroidImageonly'),'Value',0);
            data.NNonlyimage = 0;
        else
            data.NNonlylist = 0;
        end

        set(gcbf,'Userdata',data);

    case('OutputData')
        data = get(gcbf,'Userdata');

        if (data.NNonlyimage || data.NNonlylist)
            set(findobj(gcbf,'Tag','Pushbutton3'),'String','Saving');
            drawnow 
            
            [FileName,PathName] = uiputfile(fullfile(data.pathname_outfile,data.outfile),'Enter the File Root: *NearestNeighborStats.txt');
            data.pathname_outfile = PathName;
            data.outfile = FileName;
            
            data = WriteOuttoFileNNDataStatistics(data);
            set(findobj(gcbf,'Tag','Pushbutton3'),'String','Save Results');
        elseif data.isimage
            set(findobj(gcbf,'Tag','Pushbutton3'),'String','Saving');
            drawnow
            [FileName,PathName] = uiputfile(fullfile(data.pathname_outfile,data.outfile),'Enter the File Root: *NearestNeighborStats.txt');
            data.pathname_outfile = PathName;
            data.outfile = FileName;

            data = WriteOuttoFileNNDataStatistics(data);
            data = WriteOuttoFileIADataStatistics(data);
            set(findobj(gcbf,'Tag','Pushbutton3'),'String','Save Results');
        elseif data.PlanetaryData
            set(findobj(gcbf,'Tag','Pushbutton3'),'String','Saving');
            drawnow
            [FileName,PathName] = uiputfile(fullfile(data.pathname_outfile,data.outfile),'Enter the File Root: *NearestNeighborStats.txt');
            data.pathname_outfile = PathName;
            data.outfile = FileName;
            data = WriteOuttoFilePlanetNNDataStatistics(data);
            set(findobj(gcbf,'Tag','Pushbutton3'),'String','Save Results');
        end
        set(gcbf,'Userdata',data);

        %%%%%% Image Process %%%%%%%%
    case('ImageProcess')
        % This is where the data gets processed
        data = get(gcbf,'Userdata');

        if isempty(data.input1)
            set(findobj(gcbf,'Tag','Pushbutton1'),'String',' << Load IMAGE ');
            pause(1.3)
            set(findobj(gcbf,'Tag','Pushbutton1'),'String',' Process >>');
            return
        end

        set(findobj(gcbf,'Tag','Pushbutton1'),'String','please wait...');
        drawnow
        if data.NNonlyimage || data.isimage
            [data] = BubbleProcess_v02(data);
        else
            [data] = CentroidProcess_v02(data);
        end


        data.image_processed = 1;
        set(gcbf,'Userdata',data);
        set(findobj(gcbf,'Tag','Pushbutton1'),'String',' Process >>');

        dispText('','b');
        
        
% ___________--------------________________------------
        %% Bubble Frequency Controls

    case('setAxes3')
        % Change from histogram of perimeter to perimeter vs bubble equal area
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'Axes3Histogram'), 'Value')
            data.BubPerimeter = 'Histogram';
        elseif get(findobj(gcbf, 'Tag', 'Axes3Plot'), 'Value')
            data.BubPerimeter = 'Plot';
        elseif get(findobj(gcbf, 'Tag', 'Axes3RadiusPlot'), 'Value')
            data.BubPerimeter = 'RadPlot';
        else
            error('Axes3 Radion Button')
        end

        if data.image_processed
            [data] = PlotBubblePerimeters(data);
        end
        set(gcbf,'Userdata',data)

    case('setStatSummary')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'StatArea'), 'Value')
            data.SummaryStatistics = 'Area';
        elseif get(findobj(gcbf, 'Tag', 'StatPerimeter'), 'Value')
            data.SummaryStatistics = 'StatPerimeter';
        elseif get(findobj(gcbf, 'Tag', 'StatEccentricity'), 'Value')
            data.SummaryStatistics = 'StatEccentricity';
        elseif get(findobj(gcbf, 'Tag', 'StatOrientation'), 'Value')
            data.SummaryStatistics = 'StatOrientation';
        else
            error('Summary Stats Radio Button')
        end

        if data.image_processed
            [data] = WriteOutSummaryStatistic(data);
        end
        set(gcbf,'Userdata',data)

    case('setBubSizePhiCustom')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'BubSizePhi'), 'Value')  % Now a default Linear plot
            data.BubbleBinSizeOption = 'Phi';
            % reset the X axis to default, if phi mode is selected
            set(findobj(gcbf, 'Tag', 'BubXSizeDefault'), 'Value', 1)
            data.BubbleXAxisOption = 'Default';
        elseif get(findobj(gcbf, 'Tag', 'BubSizeCustom'), 'Value')
            data.BubbleBinSizeOption = 'Custom';
            t = get(findobj(gcbf,'Tag','setBubSizeCustomBin'),'String');
            data.BubbleBinSize = str2double(t);
        else
            error('Bubble Size Phi/Custom')
        end

        if data.image_processed
            [data] = PlotBubbleSizeFrequencyHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setBubSizeXCustom')
        data = get(gcbf,'Userdata');
        % If Custom X axis is set, then phi mode is disabled
        data.BubbleBinSizeOption = 'Custom';
        set(findobj(gcbf, 'Tag', 'BubXSizeCustom'), 'Value', 1)

        if get(findobj(gcbf, 'Tag', 'BubXSizeDefault'), 'Value')
            data.BubbleXAxisOption = 'Default';
        elseif get(findobj(gcbf, 'Tag', 'BubXSizeCustom'), 'Value')
            data.BubbleXAxisOption = 'Custom';
            t1 = get(findobj(gcbf,'Tag','setBubXAxisMinSize'),'String');
            t2 = get(findobj(gcbf,'Tag','setBubXAxisMaxSize'),'String');

            if isnan(str2double(t1)) && isnan(str2double(t2))
                % If Min or Max are entered, ignore the Custom command
                set(findobj(gcbf,'Tag','BubXSizeCustom'),'String','Value >');
                drawnow, pause(0.35)
                set(findobj(gcbf,'Tag','BubXSizeCustom'),'String','Custom');
                set(findobj(gcbf, 'Tag', 'BubXSizeDefault'), 'Value', 1)
                data.BubbleXAxisOption = 'Default';
            elseif isnan(str2double(t1))
                data.BubbleXAxisMinSize = 100;
                data.BubbleXAxisMaxSize = str2double(t2);
            elseif isnan(str2double(t2))
                data.BubbleXAxisMinSize = str2double(t1);
                if strcmp(data.IgnoreLargeArea, 'Max')
                    % Set the max size to the max bubble the user inputs
                    data.BubbleXAxisMaxSize = 100000;
                else
                    data.BubbleXAxisMaxSize = data.IgnoreLargeArea;
                end
            else
                data.BubbleXAxisMinSize = str2double(t1);
                data.BubbleXAxisMaxSize = str2double(t2);
                set(findobj(gcbf, 'Tag', 'BubSizeCustom'), 'Value', 1)
                data.BubbleBinSizeOption = 'Custom';
                if data.BubbleXAxisMaxSize <= data.BubbleXAxisMinSize
                    % Check max is not smaller than min!
                    set(findobj(gcbf,'Tag','BubXSizeCustom'),'String','Value >');
                    drawnow, pause(0.35)
                    set(findobj(gcbf,'Tag','BubXSizeCustom'),'String','Custom');
                    data.BubbleXAxisOption = 'Default';
                end
            end
        else
            error('Bubble X Size Default/Custom Values')
        end

        if data.image_processed
            [data] = PlotBubbleSizeFrequencyHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setBubSizeYLinLog')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'BubYSizeLinear'), 'Value')
            data.BubbleYAxisOption = 'Linear';
        elseif get(findobj(gcbf, 'Tag', 'BubYSizeLog'), 'Value')
            data.BubbleYAxisOption = 'Log';
        else
            error('Bubble Size Y Linear/Log')
        end

        if data.image_processed
            [data] = PlotBubbleSizeFrequencyHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setYAxisMaxSize')
        data = get(gcbf,'Userdata');

        t1 = get(findobj(gcbf,'Tag','YAxisMaxSizeValue'),'String');
        if strcmp(t1, 'Max') || isnan(str2double(t1))
            set(findobj(gcbf,'Tag','BubSizeYAxisMax'),'String','Value >');
            drawnow, pause(0.35)
            set(findobj(gcbf,'Tag','BubSizeYAxisMax'),'String','Custom');
            set(findobj(gcbf,'Tag','BubSizeYAxisMax'),'Value',0);
            data.BubbleYMaxAxisOption = 0;
        else
            set(findobj(gcbf,'Tag','BubSizeYAxisMax'),'Value',1);
            data.BubbleYAxisMaxSize = str2double(t1);
            data.BubbleYMaxAxisOption = 1;
        end

        if data.image_processed
            [data] = PlotBubbleSizeFrequencyHist_v01(data);
        end

        set(gcbf,'Userdata',data)

        % %  Pixel Unit selection

    case('setPixelUnitSelection')
        data = get(gcbf,'Userdata');
        t = get(findobj(gcbf,'Tag','setPixelSize'),'String');
        if isnan(str2double(t))
            set(findobj(gcbf,'Tag','setPixelSize'),'String','Value!');
            drawnow, pause(0.4)
            set(findobj(gcbf,'Tag','setPixelSize'),'String','1');
            data.PixelSize = 1;
        else
            data.PixelSize = str2double(t);
        end

        unit = get(findobj(gcbf, 'Tag', 'PixelUnits'), 'Value');
        if unit == 5
            data.PixelUnit = 'um';
            data.PixelSize = data.PixelSize;
            set(findobj(gcbf,'Tag','IAPixelUnit'),'String', 'um^2');
            set(findobj(gcbf,'Tag','NNPixelUnit'),'String', 'um');
        elseif unit == 6
            data.PixelUnit = 'um2';
            data.PixelSize = sqrt(data.PixelSize);
            set(findobj(gcbf,'Tag','IAPixelUnit'),'String', 'um^2');
            set(findobj(gcbf,'Tag','NNPixelUnit'),'String', 'um');
        elseif unit == 3
            data.PixelUnit = 'mm';
            data.PixelSize = data.PixelSize;
            set(findobj(gcbf,'Tag','IAPixelUnit'),'String', 'mm^2');
            set(findobj(gcbf,'Tag','NNPixelUnit'),'String', 'mm');
        elseif unit == 4
            data.PixelUnit = 'mm2';
            data.PixelSize = sqrt(data.PixelSize);
            set(findobj(gcbf,'Tag','IAPixelUnit'),'String', 'mm^2');
            set(findobj(gcbf,'Tag','NNPixelUnit'),'String', 'mm');
        elseif unit == 1
            data.PixelUnit = 'm';
            data.PixelSize = data.PixelSize;
            set(findobj(gcbf,'Tag','IAPixelUnit'),'String', 'm^2');
            set(findobj(gcbf,'Tag','NNPixelUnit'),'String', 'm');
        elseif unit == 2
            data.PixelUnit = 'm2';
            data.PixelSize = sqrt(data.PixelSize);
            set(findobj(gcbf,'Tag','IAPixelUnit'),'String', 'm^2');
            set(findobj(gcbf,'Tag','NNPixelUnit'),'String', 'm');
            
        else
            error('Pixel Unit Dropdown Box')
        end

        set(gcbf,'Userdata',data)

% ___________--------------________________------------
        % %  Nearest Neighbour Controls

    case('setNNModelSelection')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'Poisson'), 'Value')
            data.NNModelSelection = 'Poisson';
        elseif get(findobj(gcbf, 'Tag', 'NormPoisson'), 'Value')
            data.NNModelSelection = 'NormPoisson';
        elseif get(findobj(gcbf, 'Tag', 'ScavangedPoisson1'), 'Value')
            data.NNModelSelection = 'ScavangedPoisson';
            data.BubbleBinSize = 1;
            data.setkvalue = 1;
        elseif get(findobj(gcbf, 'Tag', 'ScavangedPoisson2'), 'Value')
            data.NNModelSelection = 'ScavangedPoisson';
            data.BubbleBinSize = 2;
            data.setkvalue = 2;

        else
            error('NN Stats Radio Button')
        end

        if data.image_processed
            [data] = PlotNNRandCgraphs(data);
            [data] = WriteOutNNStatistics(data);
        end
        set(gcbf,'Userdata',data)

    case('setNNSizeDefaultCustom')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'NNSizeDefault'), 'Value')
            data.NNBinSizeOption = 'Default';
            % reset the X axis to default, if Default mode is selected
            set(findobj(gcbf, 'Tag', 'NNXSizeDefault'), 'Value', 1)
            data.NNXAxisOption = 'Default';
        elseif get(findobj(gcbf, 'Tag', 'NNSizeCustom'), 'Value')
            data.NNBinSizeOption = 'Custom';
            t = get(findobj(gcbf,'Tag','setNNSizeCustomBin'),'String');
            data.NNBinSize = str2double(t);
        else
            error('NN Size Default/Custom')
        end

        if data.image_processed
            [data] = PlotPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setNNSizeXCustom')
        data = get(gcbf,'Userdata');
        % If Custom X axis is set, then Default mode is disabled
        data.NNBinSizeOption = 'Custom';
        set(findobj(gcbf, 'Tag', 'NNXSizeCustom'), 'Value', 1)

        if get(findobj(gcbf, 'Tag', 'NNXSizeDefault'), 'Value')
            data.NNXAxisOption = 'Default';
        elseif get(findobj(gcbf, 'Tag', 'NNXSizeCustom'), 'Value')
            data.NNXAxisOption = 'Custom';
            t1 = get(findobj(gcbf,'Tag','setNNXAxisMinSize'),'String');
            t2 = get(findobj(gcbf,'Tag','setNNXAxisMaxSize'),'String');

            if isnan(str2double(t1)) && isnan(str2double(t2))
                % If Min or Max are entered, ignore the Custom command
                set(findobj(gcbf,'Tag','NNXSizeCustom'),'String','Value >');
                drawnow, pause(0.35)
                set(findobj(gcbf,'Tag','NNXSizeCustom'),'String','Custom');
                set(findobj(gcbf, 'Tag', 'NNXSizeDefault'), 'Value', 1)
                data.NNXAxisOption = 'Default';
            elseif isnan(str2double(t1))
                data.NNXAxisMinSize = 0;
                data.NNXAxisMaxSize = str2double(t2);
            elseif isnan(str2double(t2))
                data.NNXAxisMinSize = str2double(t1);
                data.NNXAxisMaxSize = 200;
            else
                data.NNXAxisMinSize = str2double(t1);
                data.NNXAxisMaxSize = str2double(t2);
                set(findobj(gcbf, 'Tag', 'NNSizeCustom'), 'Value', 1)
                data.NNBinSizeOption = 'Custom';
                if data.NNXAxisMaxSize <= data.NNXAxisMinSize
                    % Check max is not smaller than min!
                    set(findobj(gcbf,'Tag','NNXSizeCustom'),'String','Value >');
                    drawnow, pause(0.35)
                    set(findobj(gcbf,'Tag','NNXSizeCustom'),'String','Custom');
                    data.NNXAxisOption = 'Default';
                end
            end
        else
            error('NN X Size Default/Custom Values')
        end

        if data.image_processed
            [data] = PlotPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setNNSizeYLinLog')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'NNYSizeLinear'), 'Value')
            data.NNYAxisOption = 'Linear';
        elseif get(findobj(gcbf, 'Tag', 'NNYSizeLog'), 'Value')
            data.NNYAxisOption = 'Log';
        else
            error('NN  Size Y Linear/Log')
        end

        if data.image_processed
            [data] = PlotPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setNNYAxisMaxSize')
        data = get(gcbf,'Userdata');

        t1 = get(findobj(gcbf,'Tag','NNYAxisMaxSizeValue'),'String');
        if strcmp(t1, 'Max') || isnan(str2double(t1))
            set(findobj(gcbf,'Tag','NNSizeYAxisMax'),'String','Value >');
            drawnow, pause(0.35)
            set(findobj(gcbf,'Tag','NNSizeYAxisMax'),'String','Custom');
            set(findobj(gcbf,'Tag','NNSizeYAxisMax'),'Value',0);
            data.NNYMaxAxisOption = 0;
        else
            set(findobj(gcbf,'Tag','NNSizeYAxisMax'),'Value',1);
            data.NNYAxisMaxSize = str2double(t1);
            data.NNYMaxAxisOption = 1;
        end

        if data.image_processed
            [data] = PlotPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setNNCentroidImageonly')
        data = get(gcbf,'Userdata');

        t = get(findobj(gcbf,'Tag','NNCentroidImageonly'),'Value');
        if t == 1
            data.NNonlyimage = 1;
            set(findobj(gcbf,'Tag','NNCentroidListonly'),'Value',0);
            data.NNonlylist = 0;
        else
            data.NNonlyimage = 0;
        end

        set(gcbf,'Userdata',data);

% ___________--------------________________------------

        % %  Planetary Nearest Neighbour Controls

        if data.image_processed

            [data] = PlotPlanetNNRandCgraphs(data);
            [data] = WriteOutPlanetNNStatistics(data);
        end
        set(gcbf,'Userdata',data)
        
   case('setPlanetDatasetSize')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'WholePlanet'), 'Value')
            data.PlanetDatasetSize = 'Whole';
            
        elseif get(findobj(gcbf, 'Tag', 'HalfPlanet'), 'Value')
            data.PlanetDatasetSize = 'Half';
        elseif get(findobj(gcbf, 'Tag', 'ThirdPlanet'), 'Value')
            data.PlanetDatasetSize = 'Third';
        elseif get(findobj(gcbf, 'Tag', 'QuarterPlanet'), 'Value')
            data.PlanetDatasetSize = 'Quarter';
        elseif get(findobj(gcbf, 'Tag', 'Region'), 'Value')
            data.PlanetDatasetSize = 'Regions';
        else
            error('NN Stats Radio Button')
        end

        if data.image_processed
            [data] = PlotPlanetNNRandCgraphs(data);
            [data] = WriteOutPlanetNNStatistics(data);
        end
        set(gcbf,'Userdata',data)  
        
        
    % Set the Histogram Contols
    case('setPlanetNNSizeDefaultCustom')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'PlanetNNSizeDefault'), 'Value')
            data.PlanetNNBinSizeOption = 'Default';
            % reset the X axis to default, if Default mode is selected
            set(findobj(gcbf, 'Tag', 'PlanetNNXSizeDefault'), 'Value', 1)
            data.PlanetNNXAxisOption = 'Default';
        elseif get(findobj(gcbf, 'Tag', 'PlanetNNSizeCustom'), 'Value')
            data.PlanetNNBinSizeOption = 'Custom';
            t = get(findobj(gcbf,'Tag','setPlanetNNSizeCustomBin'),'String');
            data.PlanetNNBinSize = str2double(t);
        else
            error('NN Size Default/Custom')
        end

        if data.image_processed
            [data] = PlotPlanetPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setPlanetNNSizeXCustom')
        data = get(gcbf,'Userdata');
        % If Custom X axis is set, then Default mode is disabled
        data.PlanetNNBinSizeOption = 'Custom';
        set(findobj(gcbf, 'Tag', 'PlanetNNXSizeCustom'), 'Value', 1)

        if get(findobj(gcbf, 'Tag', 'PlanetNNXSizeDefault'), 'Value')
            data.PlanetNNXAxisOption = 'Default';
        elseif get(findobj(gcbf, 'Tag', 'PlanetNNXSizeCustom'), 'Value')
            data.PlanetNNXAxisOption = 'Custom';
            t1 = get(findobj(gcbf,'Tag','setPlanetNNXAxisMinSize'),'String');
            t2 = get(findobj(gcbf,'Tag','setPlanetNNXAxisMaxSize'),'String');

            if isnan(str2double(t1)) && isnan(str2double(t2))
                % If Min or Max are entered, ignore the Custom command
                set(findobj(gcbf,'Tag','PlanetNNXSizeCustom'),'String','Value >');
                drawnow, pause(0.35)
                set(findobj(gcbf,'Tag','PlanetNNXSizeCustom'),'String','Custom');
                set(findobj(gcbf, 'Tag', 'PlanetNNXSizeDefault'), 'Value', 1)
                data.PlanetNNXAxisOption = 'Default';
            elseif isnan(str2double(t1))
                data.PlanetNNXAxisMinSize = 0;
                data.PlanetNNXAxisMaxSize = str2double(t2);
            elseif isnan(str2double(t2))
                data.PlanetNNXAxisMinSize = str2double(t1);
                data.PlanetNNXAxisMaxSize = 200;
            else
                data.PlanetNNXAxisMinSize = str2double(t1);
                data.PlanetNNXAxisMaxSize = str2double(t2);
                set(findobj(gcbf, 'Tag', 'PlanetNNSizeCustom'), 'Value', 1)
                data.PlanetNNBinSizeOption = 'Custom';
                if data.PlanetNNXAxisMaxSize <= data.PlanetNNXAxisMinSize
                    % Check max is not smaller than min!
                    set(findobj(gcbf,'Tag','PlanetNNXSizeCustom'),'String','Value >');
                    drawnow, pause(0.35)
                    set(findobj(gcbf,'Tag','PlanetNNXSizeCustom'),'String','Custom');
                    data.PlanetNNXAxisOption = 'Default';
                end
            end
        else
            error('NN X Size Default/Custom Values')
        end

        if data.image_processed
            [data] = PlotPlanetPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setPlanetNNSizeYLinLog')
        data = get(gcbf,'Userdata');

        if get(findobj(gcbf, 'Tag', 'PlanetNNYSizeLinear'), 'Value')
            data.PlanetNNYAxisOption = 'Linear';
        elseif get(findobj(gcbf, 'Tag', 'PlanetNNYSizeLog'), 'Value')
            data.PlanetNNYAxisOption = 'Log';
        else
            error('NN  Size Y Linear/Log')
        end

        if data.image_processed
            [data] = PlotPlanetPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)

    case('setPlanetNNYAxisMaxSize')
        data = get(gcbf,'Userdata');

        t1 = get(findobj(gcbf,'Tag','PlanetNNYAxisMaxSizeValue'),'String');
        if strcmp(t1, 'Max') || isnan(str2double(t1))
            set(findobj(gcbf,'Tag','PlanetNNSizeYAxisMax'),'String','Value >');
            drawnow, pause(0.35)
            set(findobj(gcbf,'Tag','PlanetNNSizeYAxisMax'),'String','Custom');
            set(findobj(gcbf,'Tag','PlanetNNSizeYAxisMax'),'Value',0);
            data.NNYMaxAxisOption = 0;
            data.PlanetNNYAxisMaxSize = Inf;
        else
            set(findobj(gcbf,'Tag','PlanetNNSizeYAxisMax'),'Value',1);
            data.PlanetNNYAxisMaxSize = str2double(t1);
            data.PlanetNNYMaxAxisOption = 1;
        end

        if data.image_processed
            [data] = PlotPlanetPoissonNNHist_v01(data);
        end

        set(gcbf,'Userdata',data)


    otherwise
        disp(['gias_guifun.m: trying to call non-existent switch...',action])
end



% *************************** auxilliary functions

function dispText(txt,colr)
set(findobj(gcbf,'Tag','MessText'),'String',txt,'ForegroundColor',colr);

function dispTag(txt,colr)


set(findobj(gcbf,'Tag',txt),'ForegroundColor',colr);